/*
 * Copyright (c) 2015, 2016, Oracle and/or its affiliates. All rights reserved.
 * ORACLE PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 *
 */

package com.sun.tools.javac.util;

import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.function.Function;
import java.util.function.Predicate;

/** Utilities for Iterators.
 *
 *  <p><b>This is NOT part of any supported API.
 *  If you write code that depends on this, you do so at your own risk.
 *  This code and its internal interfaces are subject to change or
 *  deletion without notice.</b>
 */
public class Iterators {

    public static <I, O> Iterator<O> createCompoundIterator(Iterable<I> inputs, Function<I, Iterator<O>> converter) {
        return new CompoundIterator<>(inputs, converter);
    }

    private static class CompoundIterator<I, O> implements Iterator<O> {

        private final Iterator<I> inputs;
        private final Function<I, Iterator<O>> converter;
        @SuppressWarnings("unchecked")
        private Iterator<O> currentIterator = EMPTY;

        public CompoundIterator(Iterable<I> inputs, Function<I, Iterator<O>> converter) {
            this.inputs = inputs.iterator();
            this.converter = converter;
        }

        public boolean hasNext() {
            if (currentIterator != null && !currentIterator.hasNext()) {
                update();
            }
            return currentIterator != null;
        }

        public O next() {
            if (currentIterator == EMPTY && !hasNext()) {
                throw new NoSuchElementException();
            }
            return currentIterator.next();
        }

        public void remove() {
            throw new UnsupportedOperationException();
        }

        private void update() {
            while (inputs.hasNext()) {
                currentIterator = converter.apply(inputs.next());
                if (currentIterator.hasNext()) return;
            }
            currentIterator = null;
        }
    }

    @SuppressWarnings("rawtypes")
    private static final Iterator EMPTY = new Iterator() {
        public boolean hasNext() {
            return false;
        }

        @Override
        public Object next() {
            return null;
        }
    };

    public static <E> Iterator<E> createFilterIterator(Iterator<E> input, Predicate<E> test) {
        return new Iterator<E>() {
            private E current = update();
            private E update () {
                while (input.hasNext()) {
                    E sym = input.next();
                    if (test.test(sym)) {
                        return sym;
                    }
                }

                return null;
            }
            @Override
            public boolean hasNext() {
                return current != null;
            }

            @Override
            public E next() {
                E res = current;
                current = update();
                return res;
            }
        };
    }

}
